#!/usr/bin/env python3
# A3 Monotone Ladder Audit — self-contained engine (stdlib only)
import argparse, csv, hashlib, json, math, os, sys, time
from pathlib import Path

# ---------- utils ----------
def ensure_dir(p: Path): p.mkdir(parents=True, exist_ok=True)
def sha256_of_file(p: Path):
    h = hashlib.sha256()
    with p.open('rb') as f:
        for chunk in iter(lambda: f.read(1<<20), b''):
            h.update(chunk)
    return h.hexdigest()
def sha256_of_text(s: str): return hashlib.sha256(s.encode('utf-8')).hexdigest()
def write_json(p: Path, obj): ensure_dir(p.parent); p.write_text(json.dumps(obj, indent=2), encoding='utf-8')
def write_csv(p: Path, header, rows):
    ensure_dir(p.parent)
    with p.open('w', newline='', encoding='utf-8') as f:
        w = csv.writer(f); w.writerow(header); w.writerows(rows)

def load_json(p: Path, default=None):
    if p is None: return default
    if not p.exists(): return default
    return json.loads(p.read_text(encoding='utf-8'))

# ---------- audit helpers ----------
def is_monotone_inward(strict):
    """
    Shell indexing: 0 = INNERMOST, K-1 = OUTERMOST.
    Monotone 'inward' means strictness does NOT ease toward the center:
    s[0] >= s[1] >= ... >= s[K-1]  (non-increasing outward).
    """
    viol = []
    for k in range(len(strict)-1):
        if strict[k] < strict[k+1]:
            viol.append((k, k+1))
    return (len(viol) == 0), viol

def build_acceptance_profile(strict):
    """
    Map integer strictness to a deterministic acceptance probability p_k,
    strictly non-decreasing outward (i.e., inner <= outer), to mirror 'more strict inward'.
    """
    smin, smax = min(strict), max(strict)
    if smax == smin:
        return [0.60 for _ in strict]  # uniform acceptance if all strictness equal
    # linear map: inner (higher strict) gets smaller p
    p = []
    for s in strict:
        x = (s - smin) / (smax - smin)        # 0..1
        # inner high-s -> smaller p; clamp to [0.30, 0.85]
        p_k = 0.85 - 0.55 * x
        if p_k < 0.30: p_k = 0.30
        if p_k > 0.85: p_k = 0.85
        p.append(p_k)
    return p

# ---------- main ----------
def main():
    ap = argparse.ArgumentParser()
    ap.add_argument('--manifest', required=True)   # JSON
    ap.add_argument('--diag', required=True)       # JSON (tolerances, attempts)
    ap.add_argument('--out', required=True)        # output directory
    args = ap.parse_args()

    out_dir = Path(args.out)
    metrics_dir = out_dir/'metrics'
    runinfo_dir = out_dir/'run_info'
    audits_dir = out_dir/'audits'
    for d in [metrics_dir, runinfo_dir, audits_dir]:
        ensure_dir(d)

    manifest_path = Path(args.manifest)
    diag_path = Path(args.diag)
    if not manifest_path.exists(): raise FileNotFoundError(f"Manifest not found: {manifest_path}")
    if not diag_path.exists(): raise FileNotFoundError(f"Diagnostics config not found: {diag_path}")

    manifest = load_json(manifest_path)
    diag = load_json(diag_path, {})

    # hashes/provenance
    manifest_hash = sha256_of_file(manifest_path)
    measure_hash = sha256_of_text("Haar_unit_circle|counting")
    hinge_hash = sha256_of_text("A3:schedule=ON;monotone_ladder_audit")

    # read config
    H = int(manifest.get('domain',{}).get('ticks',128))
    strict = manifest.get('engine_contract',{}).get('strictness_by_shell', [3,2,2,1])
    K = len(strict)
    schedule = manifest.get('engine_contract',{}).get('schedule', 'ON')

    tau_mono = float(diag.get('tolerances',{}).get('tau_mono', 0.0))
    attempts_per_tick = int(diag.get('attempts',{}).get('per_shell_per_tick', 2000))
    # deterministic acceptance (no RNG) to avoid false flips
    deterministic = True

    # 1) Declaration check (monotone vector)
    decl_ok, decl_viol = is_monotone_inward(strict)

    # 2) Observed check via a simple availability model
    # acceptance probability per shell, inward stricter => lower p
    p = build_acceptance_profile(strict)

    # Deterministic counts per shell
    attempts_by_shell = [H * attempts_per_tick for _ in range(K)]
    accepts_by_shell = [int(round(attempts_by_shell[k] * p[k])) for k in range(K)]
    rates_by_shell = [ (accepts_by_shell[k] / attempts_by_shell[k]) if attempts_by_shell[k] > 0 else 0.0
                       for k in range(K) ]

    # observed monotonicity: inner rate <= outer rate (within tau_mono)
    obs_viol = []
    for k in range(K-1):
        if rates_by_shell[k] > rates_by_shell[k+1] + tau_mono:
            obs_viol.append((k, k+1, rates_by_shell[k], rates_by_shell[k+1]))

    V = len(obs_viol)
    PASS = (schedule == 'ON') and decl_ok and (V == 0)

    # metrics CSV
    rows = []
    for k in range(K):
        rows.append([
            k, strict[k],
            attempts_by_shell[k], accepts_by_shell[k],
            round(rates_by_shell[k], 8), round(p[k], 8)
        ])
    write_csv(
        metrics_dir/'acceptance_by_shell.csv',
        ['shell_id','strictness','attempts','accepts','accept_rate','p_design'],
        rows
    )

    # audit JSON
    write_json(
        audits_dir/'monotone_schedule.json',
        {
            "schedule": schedule,
            "strictness_by_shell": strict,
            "declared_monotone_inward": decl_ok,
            "declared_violations": [{"pair": list(v)} for v in decl_viol],
            "observed_accept_rates": rates_by_shell,
            "observed_violations": [
                {"pair":[a,b], "inner_rate":ri, "outer_rate":ro} for (a,b,ri,ro) in obs_viol
            ],
            "tau_mono": tau_mono,
            "V": V,
            "PASS": PASS
        }
    )

    # hashes & provenance
    write_json(
        runinfo_dir/'hashes.json',
        {
            "manifest_hash": manifest_hash,
            "measure_hash": measure_hash,
            "hinge_hash": hinge_hash,
            "engine_entrypoint": f"python {Path(sys.argv[0]).name} --manifest <...> --diag <...> --out <...>"
        }
    )

    # stdout summary
    summary = {
        "declared_monotone": decl_ok,
        "max_observed_rate": max(rates_by_shell) if rates_by_shell else None,
        "min_observed_rate": min(rates_by_shell) if rates_by_shell else None,
        "V": V,
        "PASS": PASS,
        "audit_path": str((audits_dir/'monotone_schedule.json').as_posix())
    }
    print("A3 SUMMARY:", json.dumps(summary))

if __name__ == '__main__':
    try:
        main()
    except Exception as e:
        # explicit failure with reason
        try:
            out_dir = None
            for i,a in enumerate(sys.argv):
                if a == '--out' and i+1 < len(sys.argv): out_dir = Path(sys.argv[i+1])
            if out_dir:
                audits = out_dir/'audits'; ensure_dir(audits)
                write_json(audits/'monotone_schedule.json',
                           {"PASS": False, "failure_reason": f"Unexpected error: {type(e).__name__}: {e}"})
        finally:
            raise
